<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_REST { // OAI-260103: HubWright REST surface.
	private $settings;

	public function __construct( HW_Settings $settings ) {
		$this->settings = $settings;
	}

	public function register_routes() {
		add_action(
			'rest_api_init',
			function () {
				register_rest_route(
					'hw/v1',
					'/token',
					array(
						'methods'             => WP_REST_Server::READABLE,
						'callback'            => array( $this, 'issue_token' ),
						'permission_callback' => function () {
							return is_user_logged_in();
						},
					)
				);
			}
		);

		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	public function issue_token() {
		$current_user = wp_get_current_user();
		$options      = $this->settings->get_options();

		if ( empty( $options['site_secret'] ) || empty( $options['site_key'] ) ) {
			return new WP_Error( 'hw_missing_config', 'Hub keys are not configured.', array( 'status' => 400 ) );
		}

		$token = $this->generate_jwt(
			array(
				'sub'     => (string) $current_user->ID,
				'site_id' => $options['site_key'],
				'roles'   => $current_user->roles,
				'iat'     => time(),
				'exp'     => time() + 900, // 15 minutes.
			),
			$options['site_secret']
		);

		return rest_ensure_response(
			array(
				'token' => $token,
			)
		);
	}

	private function generate_jwt( $payload, $secret ) {
		// Minimal JWT signer for MVP; replace with a vetted JWT lib before production.
		$header       = array( 'alg' => 'HS256', 'typ' => 'JWT' );
		$segments     = array();
		$segments[]   = $this->base64url_encode( wp_json_encode( $header ) );
		$segments[]   = $this->base64url_encode( wp_json_encode( $payload ) );
		$signing_input = implode( '.', $segments );
		$signature    = hash_hmac( 'sha256', $signing_input, $secret, true );
		$segments[]   = $this->base64url_encode( $signature );

		return implode( '.', $segments );
	}

	private function base64url_encode( $data ) {
		return rtrim( strtr( base64_encode( $data ), '+/', '-_' ), '=' );
	}

	public function enqueue_assets() {
		wp_enqueue_script(
			'hw-app',
			HW_URL . 'assets/js/app.js',
			array( 'wp-element', 'wp-api-fetch' ),
			HW_VERSION,
			true
		);

		wp_localize_script(
			'hw-app',
			'hwData',
			array(
				'hubUrl'      => $this->settings->get_options()['hub_url'],
				'tokenRoute'  => rest_url( 'hw/v1/token' ),
				'currentUser' => get_current_user_id(),
			)
		);
	}
}
