<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_Paywall { // OAI-260103: Paywall gating and post-level toggle.
	private $settings;
	private $meta_key = '_hubw_paywall_gate';
	private $share_meta = 'hubw_share_via_hub'; // OAI-260104: per-post share toggle.
	private $tier_meta  = 'hubw_tier_level'; // OAI-260104: per-post tier selection.

	public function __construct( HW_Settings $settings ) {
		$this->settings = $settings;
	}

	public function register() {
		add_action( 'init', array( $this, 'register_meta' ) );
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
		add_action( 'save_post', array( $this, 'save_meta' ) );
		add_filter( 'the_content', array( $this, 'filter_content' ), 9 );
		add_shortcode( 'hw_gate', array( $this, 'render_shortcode' ) );
		add_action( 'init', array( $this, 'register_block' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
	}

	public function register_meta() {
		register_post_meta(
			'post',
			$this->share_meta,
			array(
				'type'         => 'boolean',
				'single'       => true,
				'show_in_rest' => true,
				'auth_callback' => function() {
					return current_user_can( 'edit_posts' );
				},
			)
		);
		register_post_meta(
			'post',
			$this->tier_meta,
			array(
				'type'         => 'string',
				'single'       => true,
				'show_in_rest' => true,
				'auth_callback' => function() {
					return current_user_can( 'edit_posts' );
				},
			)
		);
	}

	public function add_meta_box() {
		add_meta_box(
			'hw_paywall',
			'hubWright Paywall',
			array( $this, 'render_meta_box' ),
			'post',
			'side',
			'default'
		);
	}

	public function render_meta_box( $post ) {
		wp_nonce_field( 'hw_paywall_nonce', 'hw_paywall_nonce_field' );
		$enabled = get_post_meta( $post->ID, $this->meta_key, true );
		if ( '' === $enabled ) {
			$enabled = get_post_meta( $post->ID, '_hw_paywall_gate', true ); // OAI-260104: legacy meta fallback.
		}
		$share    = get_post_meta( $post->ID, $this->share_meta, true );
		$tier     = get_post_meta( $post->ID, $this->tier_meta, true );
		$options  = $this->settings->get_options();
		$tiers    = $this->settings->get_tier_names( $options );
		if ( ! $tier ) {
			$tier = 'Free';
		}
		?>
		<p>
			<label>
				<input type="checkbox" name="hw_paywall_enabled" value="1" <?php checked( $enabled, '1' ); ?> />
				Protect this post with the hubWright paywall.
			</label>
		</p>
		<p>
			<label>
				<input type="checkbox" name="hw_share_via_hub" value="1" <?php checked( $share, '1' ); ?> />
				Share via hub
			</label>
		</p>
		<p>
			<label for="hw_tier_level">Tier</label><br>
			<select name="hw_tier_level" id="hw_tier_level">
				<?php foreach ( $tiers as $tier_option ) : ?>
					<option value="<?php echo esc_attr( $tier_option ); ?>" <?php selected( $tier_option, $tier ); ?>><?php echo esc_html( $tier_option ); ?></option>
				<?php endforeach; ?>
			</select>
		</p>
		<?php
	}

	public function save_meta( $post_id ) {
		if ( ! isset( $_POST['hw_paywall_nonce_field'] ) || ! wp_verify_nonce( $_POST['hw_paywall_nonce_field'], 'hw_paywall_nonce' ) ) {
			return;
		}
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}
		$enabled = isset( $_POST['hw_paywall_enabled'] ) ? '1' : '0';
		update_post_meta( $post_id, $this->meta_key, $enabled );
		delete_post_meta( $post_id, '_hw_paywall_gate' ); // OAI-260104: clean legacy meta.
		$share = isset( $_POST['hw_share_via_hub'] ) ? '1' : '0';
		update_post_meta( $post_id, $this->share_meta, $share );
		if ( isset( $_POST['hw_tier_level'] ) ) {
			$tier = sanitize_text_field( wp_unslash( $_POST['hw_tier_level'] ) );
			$options = $this->settings->get_options();
			$tiers   = isset( $options['tiers'] ) && is_array( $options['tiers'] ) ? $options['tiers'] : array( 'Free' );
			if ( ! in_array( $tier, $tiers, true ) ) {
				$tier = 'Free';
			}
			update_post_meta( $post_id, $this->tier_meta, $tier );
		}
	}

	public function filter_content( $content ) {
		if ( is_admin() || ! is_singular( 'post' ) ) {
			return $content;
		}

		global $post;
		if ( ! $post ) {
			return $content;
		}

		if ( ! $this->should_gate( $post ) ) {
			return $content;
		}

		if ( is_user_logged_in() ) {
			return $content;
		}

		$options = $this->settings->get_options();
		return $this->render_paywall_notice( $post, $content, $options );
	}

	private function should_gate( $post ) {
		$options = $this->settings->get_options();
		if ( empty( $options['enable_paywall'] ) ) {
			return false;
		}

		$meta = get_post_meta( $post->ID, $this->meta_key, true );
		if ( '' === $meta ) {
			$meta = get_post_meta( $post->ID, '_hw_paywall_gate', true ); // OAI-260104: legacy meta fallback.
		}
		$gate = ( $meta === '1' );

		/**
		 * Filter whether hubWright should gate this post.
		 *
		 * @param bool   $gate Whether gating is active.
		 * @param WP_Post $post Post object.
		 */
		return (bool) apply_filters( 'hw_should_gate_post', $gate, $post );
	}

	// OAI-260103: Render gated payload for filters/shortcodes/blocks.
	private function render_paywall_notice( $post, $content, $options ) {
		$message     = ! empty( $options['paywall_message'] ) ? wp_kses_post( $options['paywall_message'] ) : 'This post is behind the paywall. Please sign in to continue.'; // OAI-260103: default paywall message.
		$show_excerpt = ! empty( $options['paywall_show_excerpt'] );
		$excerpt      = '';

		if ( $show_excerpt ) {
			if ( has_excerpt( $post ) ) {
				$excerpt = wpautop( get_the_excerpt( $post ) );
			} else {
				$excerpt = wpautop( wp_trim_words( wp_strip_all_tags( $post->post_content ), 40, '&hellip;' ) );
			}
		}

		$login_url = wp_login_url( get_permalink( $post ) );
		$hub_url   = ! empty( $options['hub_url'] ) ? esc_url( trailingslashit( $options['hub_url'] ) . 'my' ) : '';

		$notice  = '<div class="hw-paywall" style="border:1px solid #e2e8f0;padding:16px;border-radius:8px;background:#f8fafc;">';
		if ( $excerpt ) {
			$notice .= '<div class="hw-paywall-excerpt" style="margin-bottom:12px;color:#0f172a;">' . $excerpt . '</div>';
		}
		$notice .= '<div class="hw-paywall-message" style="margin-bottom:8px;font-weight:600;color:#0f172a;">' . $message . '</div>';
		$notice .= '<div class="hw-paywall-actions" style="display:flex;gap:8px;flex-wrap:wrap;">';
		$notice .= '<a class="button button-primary" href="' . esc_url( $login_url ) . '">Sign in to continue</a>';
		if ( $hub_url ) {
			$notice .= '<a class="button" href="' . $hub_url . '">Go to WordierPress</a>';
		}
		$notice .= '</div></div>';

		return $notice;
	}

	// OAI-260103: Shortcode for partial gating in classic editor: [hw_gate]secret[/hw_gate].
	public function render_shortcode( $atts, $content = '' ) {
		global $post;
		if ( ! $post ) {
			return $content;
		}

		$options = $this->settings->get_options();
		if ( empty( $options['enable_paywall'] ) ) {
			return $content;
		}
		if ( is_user_logged_in() ) {
			return do_shortcode( $content );
		}

		return $this->render_paywall_notice( $post, $content, $options );
	}

	// OAI-260103: Gutenberg block registration for gated sections.
	public function register_block() {
		register_block_type(
			'hubwright/gated-content',
			array(
				'render_callback' => array( $this, 'render_block' ),
				'supports'        => array(
					'html' => false,
				),
			)
		);
	}

	public function render_block( $attributes, $content = '' ) {
		global $post;
		if ( ! $post ) {
			return $content;
		}

		$options = $this->settings->get_options();
		if ( empty( $options['enable_paywall'] ) ) {
			return $content;
		}
		if ( is_user_logged_in() ) {
			return $content;
		}

		return $this->render_paywall_notice( $post, $content, $options );
	}

	public function enqueue_block_editor_assets() {
		wp_enqueue_script(
			'hw-gated-block',
			HW_URL . 'assets/js/gated-block.js',
			array( 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n', 'wp-data', 'wp-plugins', 'wp-edit-post' ),
			HW_VERSION,
			true
		);
		// Merge paywall toggle into WordierPress document panel (inlined for simplicity). // OAI-260104
		$script = <<<JS
( function( wp ) {
	if ( ! wp || ! wp.plugins || ! wp.editPost || ! wp.components || ! wp.data || ! wp.element ) {
		return;
	}
	const { registerPlugin } = wp.plugins;
	const { PluginDocumentSettingPanel } = wp.editPost;
	const { CheckboxControl } = wp.components;
	const { useSelect, useDispatch } = wp.data;
	const { createElement: el } = wp.element;
	const META_PAYWALL = '_hubw_paywall_gate';
	const META_AUTHORI = 'hubw_authori_queue';
	const META_SHARE = 'hubw_share_via_hub';
	let registered = false;

	const Panel = () => {
		const meta = useSelect( ( select ) => select( 'core/editor' ).getEditedPostAttribute( 'meta' ) || {}, [] );
		const paywallValue = meta[ META_PAYWALL ] === '1' || meta[ META_PAYWALL ] === true;
		const authoriValue = !! meta[ META_AUTHORI ];
		const shareValue = !! meta[ META_SHARE ];
		const { editPost } = useDispatch( 'core/editor' );

		const setMeta = ( key, val ) => {
			editPost( { meta: { ...meta, [ key ]: val } } );
		};

		return el(
			PluginDocumentSettingPanel,
			{
				name: 'hw-wordierpress-panel',
				title: 'WordierPress',
				className: 'hw-wordierpress-panel',
				style: {
					background: '#ccccff',
					border: '1px solid #cbd5e1',
				},
			},
			el( CheckboxControl, {
				label: 'Protect this post with the hubWright paywall',
				checked: paywallValue,
				onChange: ( next ) => setMeta( META_PAYWALL, next ? '1' : '0' ),
			} ),
			el( CheckboxControl, {
				label: 'Share via hub',
				checked: shareValue,
				onChange: ( next ) => setMeta( META_SHARE, next ),
			} ),
			el( CheckboxControl, {
				label: 'Copyright this content in Authori ledger',
				checked: authoriValue,
				onChange: ( next ) => setMeta( META_AUTHORI, next ),
			} )
		);
	};

	if ( ! registered ) {
		registerPlugin( 'hw-wordierpress-panel', { render: Panel } );
		registered = true;
	}
} )( window.wp );
JS;
		wp_add_inline_script( 'hw-gated-block', $script );
	}
}
