<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_Admin {
	public function register() {
		add_action( 'admin_menu', array( $this, 'add_menu' ) );
		add_filter( 'manage_posts_columns', array( $this, 'add_post_columns' ) );
		add_action( 'manage_posts_custom_column', array( $this, 'render_post_columns' ), 10, 2 );
		add_action( 'admin_post_hw_toggle_share', array( $this, 'handle_toggle_share' ) );
		add_action( 'admin_post_hw_toggle_authori', array( $this, 'handle_toggle_authori' ) );
		add_action( 'admin_post_hw_set_tier', array( $this, 'handle_set_tier' ) );
	}

	public function add_menu() {
		add_menu_page(
			'hubWright',
			'hubWright',
			'manage_options',
			'hw-admin',
			array( $this, 'render_page_settings' ),
			'dashicons-randomize',
			61 // OAI-260103: place near top-level items.
		);

		// OAI-260104: keep submenu and tabs in strict A–Z order.
		add_submenu_page( 'hw-admin', 'About', 'About', 'manage_options', 'hw-admin-about', array( $this, 'render_page_about' ) );
		add_submenu_page( 'hw-admin', 'Access', 'Access', 'manage_options', 'hw-admin', array( $this, 'render_page_settings' ) );
		add_submenu_page( 'hw-admin', 'Authori', 'Authori', 'manage_options', 'hw-admin-authori', array( $this, 'render_page_authori' ) );
		add_submenu_page( 'hw-admin', 'Backups', 'Backups', 'manage_options', 'hw-admin-backups', array( $this, 'render_page_backups' ) );
		add_submenu_page( 'hw-admin', 'Fans', 'Fans', 'manage_options', 'hw-admin-fans', array( $this, 'render_page_fans' ) );
		add_submenu_page( 'hw-admin', 'Import', 'Import', 'manage_options', 'hw-admin-substack', array( $this, 'render_page_substack' ) );
		add_submenu_page( 'hw-admin', 'Income', 'Income', 'manage_options', 'hw-admin-income', array( $this, 'render_page_income' ) );
		add_submenu_page( 'hw-admin', 'Kiosks', 'Kiosks', 'manage_options', 'hw-admin-kiosks', array( $this, 'render_page_kiosks' ) );
		add_submenu_page( 'hw-admin', 'Settings', 'Settings', 'manage_options', 'hw-admin-settings', array( $this, 'render_page_settings' ) );
		add_submenu_page( 'hw-admin', 'Share', 'Share', 'manage_options', 'hw-admin-share', array( $this, 'render_page_share' ) );
		add_submenu_page( 'hw-admin', 'Sys', 'Sys', 'manage_options', 'hw-admin-sys', array( $this, 'render_page_sys' ) );
		add_submenu_page( 'hw-admin', 'Tiers', 'Tiers', 'manage_options', 'hw-admin-paywall', array( $this, 'render_page_paywall' ) );
		add_submenu_page( 'hw-admin', 'Update', 'Update', 'manage_options', 'hw-admin-update', array( $this, 'render_page_update' ) );
	}

	public function render_page_settings() {
		$this->render_page( 'settings' );
	}

	public function render_page_update() {
		$this->render_page( 'update' );
	}

	public function render_page_fans() {
		$this->render_page( 'fans' );
	}

	public function render_page_income() {
		$this->render_page( 'income' );
	}

	public function render_page_kiosks() {
		$this->render_page( 'kiosks' );
	}

	public function render_page_paywall() {
		$this->render_page( 'paywall' );
	}

	public function render_page_backups() {
		$this->render_page( 'backups' );
	}

	public function render_page_share() {
		$this->render_page( 'share' );
	}

	public function render_page_substack() {
		$this->render_page( 'substack' );
	}

	public function render_page_sys() {
		$this->render_page( 'sys' );
	}

	public function render_page_authori() {
		$this->render_page( 'authori' );
	}

	public function add_post_columns( $columns ) {
		$columns['hw_share']   = 'Share via hub';
		$columns['hw_authori'] = 'Authori';
		$columns['hw_tier']    = 'Tier';
		return $columns;
	}

	public function render_post_columns( $column, $post_id ) {
		$options = get_option( 'hubw_settings', array() );
		if ( empty( $options ) ) {
			$options = get_option( 'hw_settings', array() );
		}
		$settings = new HW_Settings();
		$tiers    = $settings->get_tier_names( $options );
		if ( $column === 'hw_share' ) {
			$share = get_post_meta( $post_id, 'hubw_share_via_hub', true );
			?>
			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<?php wp_nonce_field( 'hw_toggle_share_' . $post_id ); ?>
				<input type="hidden" name="action" value="hw_toggle_share" />
				<input type="hidden" name="post_id" value="<?php echo esc_attr( $post_id ); ?>" />
				<label><input type="checkbox" name="share" value="1" <?php checked( $share, 1 ); ?> /> Share</label>
				<button class="button button-small" type="submit">Save</button>
			</form>
			<?php
		} elseif ( $column === 'hw_authori' ) {
			$stamped = get_post_meta( $post_id, '_hubw_authori_stamped_at', true );
			$queued  = get_post_meta( $post_id, 'hubw_authori_queue', true );
			if ( $stamped ) {
				echo '<span style="color:green;font-weight:600;">Stamped</span><br><small>' . esc_html( $stamped ) . '</small>';
			} else {
				?>
				<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
					<?php wp_nonce_field( 'hw_toggle_authori_' . $post_id ); ?>
					<input type="hidden" name="action" value="hw_toggle_authori" />
					<input type="hidden" name="post_id" value="<?php echo esc_attr( $post_id ); ?>" />
					<label><input type="checkbox" name="authori" value="1" <?php checked( $queued, 1 ); ?> /> Queue Authori</label>
					<button class="button button-small" type="submit">Save</button>
				</form>
				<?php
			}
		} elseif ( $column === 'hw_tier' ) {
			$current = get_post_meta( $post_id, 'hubw_tier_level', true );
			if ( ! $current ) {
				$current = 'Free';
			}
			?>
			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<?php wp_nonce_field( 'hw_set_tier_' . $post_id ); ?>
				<input type="hidden" name="action" value="hw_set_tier" />
				<input type="hidden" name="post_id" value="<?php echo esc_attr( $post_id ); ?>" />
				<select name="tier">
					<?php foreach ( $tiers as $tier ) : ?>
						<option value="<?php echo esc_attr( $tier ); ?>" <?php selected( $tier, $current ); ?>><?php echo esc_html( $tier ); ?></option>
					<?php endforeach; ?>
				</select>
				<button class="button button-small" type="submit">Save</button>
			</form>
			<?php
		}
	}

	public function handle_toggle_share() {
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) || ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'hw_toggle_share_' . $post_id ) ) {
			wp_die( 'Invalid request' );
		}
		$share = isset( $_POST['share'] ) ? '1' : '0';
		update_post_meta( $post_id, 'hubw_share_via_hub', $share );
		wp_safe_redirect( wp_get_referer() ? wp_get_referer() : admin_url( 'edit.php' ) );
		exit;
	}

	public function handle_toggle_authori() {
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) || ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'hw_toggle_authori_' . $post_id ) ) {
			wp_die( 'Invalid request' );
		}
		$authori = isset( $_POST['authori'] ) ? '1' : '0';
		update_post_meta( $post_id, 'hubw_authori_queue', $authori );
		wp_safe_redirect( wp_get_referer() ? wp_get_referer() : admin_url( 'edit.php' ) );
		exit;
	}

	public function handle_set_tier() {
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) || ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'hw_set_tier_' . $post_id ) ) {
			wp_die( 'Invalid request' );
		}
		$tier = isset( $_POST['tier'] ) ? sanitize_text_field( wp_unslash( $_POST['tier'] ) ) : 'Free';
		$options = get_option( 'hubw_settings', array( 'tiers' => array( 'Free' ) ) );
		if ( ! in_array( $tier, $options['tiers'], true ) ) {
			$tier = 'Free';
		}
		update_post_meta( $post_id, 'hubw_tier_level', $tier );
		wp_safe_redirect( wp_get_referer() ? wp_get_referer() : admin_url( 'edit.php' ) );
		exit;
	}
	public function render_page_about() {
		$this->render_page( 'about' );
	}

	public function render_page( $forced_tab = 'settings' ) {
		global $parent_file, $submenu_file, $wpdb;
		$parent_file  = 'hw-admin'; // OAI-260103: keep hubWright highlighted.
		$submenu_file = 'hw-admin';

		$backups       = $this->list_backups();
		$pushes        = $this->list_push_log();
		$authori       = class_exists( 'HW_Authori' ) ? ( new HW_Authori( new HW_Settings() ) )->list_log_entries() : array();
		$archives      = function_exists( 'hw_get_update_archives' ) ? hw_get_update_archives() : array();
		// Simplified update flow: always pull latest package, no version gating.
		$plugin_size    = $this->get_directory_size( WP_PLUGIN_DIR . '/hubwright' );
		$archive_size   = $this->get_directory_size( trailingslashit( WP_CONTENT_DIR ) . 'hw-updates' );
		$update_status = isset( $_GET['hw_update'] ) ? sanitize_text_field( wp_unslash( $_GET['hw_update'] ) ) : '';
		$update_msg    = isset( $_GET['hw_msg'] ) ? sanitize_text_field( wp_unslash( $_GET['hw_msg'] ) ) : '';
		$update_prev   = isset( $_GET['hw_prev'] ) ? sanitize_text_field( wp_unslash( $_GET['hw_prev'] ) ) : '';
		$authori_status = isset( $_GET['hw_authori'] ) ? sanitize_text_field( wp_unslash( $_GET['hw_authori'] ) ) : '';
		$authori_msg    = isset( $_GET['hw_msg'] ) ? sanitize_text_field( wp_unslash( $_GET['hw_msg'] ) ) : '';
		$tab_param     = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
		$tab           = $tab_param ? $tab_param : $forced_tab;
		$options       = get_option( 'hubw_settings', array() );
		if ( empty( $options ) ) {
			$options = get_option( 'hw_settings', array() ); // legacy fallback.
		}
		if ( empty( $options['tiers'] ) || ! is_array( $options['tiers'] ) ) {
			$options['tiers'] = array(
				array(
					'name'  => 'Free',
					'desc'  => '',
					'price' => '',
				),
			);
		}
		?>
		<div class="wrap">
			<h1>hubWright Admin</h1>
			<p>hubWright, the WordPress tool for members of WordierPress, the coalition of copyrighted content creators.</p>

			<h2 class="nav-tab-wrapper" style="margin-bottom:16px;">
				<a class="nav-tab <?php echo $tab === 'about' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=about' ) ); ?>">About</a>
				<a class="nav-tab <?php echo $tab === 'access' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=access' ) ); ?>">Access</a>
				<a class="nav-tab <?php echo $tab === 'authori' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=authori' ) ); ?>">Authori</a>
				<a class="nav-tab <?php echo $tab === 'backups' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=backups' ) ); ?>">Backups</a>
				<a class="nav-tab <?php echo $tab === 'fans' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=fans' ) ); ?>">Fans</a>
				<a class="nav-tab <?php echo $tab === 'substack' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=substack' ) ); ?>">Import</a>
				<a class="nav-tab <?php echo $tab === 'income' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=income' ) ); ?>">Income</a>
				<a class="nav-tab <?php echo $tab === 'kiosks' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=kiosks' ) ); ?>">Kiosks</a>
				<a class="nav-tab <?php echo $tab === 'settings' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=settings' ) ); ?>">Settings</a>
				<a class="nav-tab <?php echo $tab === 'share' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=share' ) ); ?>">Share</a>
				<a class="nav-tab <?php echo $tab === 'sys' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=sys' ) ); ?>">Sys</a>
				<a class="nav-tab <?php echo $tab === 'paywall' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=paywall' ) ); ?>">Tiers</a>
				<a class="nav-tab <?php echo $tab === 'update' ? 'nav-tab-active' : ''; ?>" href="<?php echo esc_url( admin_url( 'admin.php?page=hw-admin&tab=update' ) ); ?>">Update</a>
			</h2>

			<?php if ( $update_status === 'success' ) : ?>
				<div class="notice notice-success"><p>hubWright updated and activated.</p></div>
			<?php elseif ( $update_status === 'fail' ) : ?>
				<div class="notice notice-error">
					<p>
						<?php
						printf(
							/* translators: 1: error message, 2: version string */
							esc_html__( 'Update failed: %1$s Reverted to previous version: %2$s', 'hubwright' ),
							esc_html( $update_msg ),
							esc_html( $update_prev ? $update_prev : HW_VERSION )
						);
						?>
					</p>
				</div>
			<?php elseif ( $update_status === 'none' ) : ?>
				<div class="notice notice-info"><p>No newer version found.</p></div>
			<?php endif; ?>

			<?php if ( $authori_status === 'stamped' ) : ?>
				<div class="notice notice-success"><p><?php echo esc_html( $authori_msg ? $authori_msg : 'Stamped to Authori.' ); ?></p></div>
			<?php elseif ( $authori_status === 'fail' ) : ?>
				<div class="notice notice-error"><p><?php echo esc_html( $authori_msg ? $authori_msg : 'Authori stamp failed.' ); ?></p></div>
			<?php endif; ?>

			<?php if ( $tab === 'settings' ) : ?>
				<div id="hw-settings" class="card">
					<h2>Settings</h2>
					<p>Hub URL: <code><?php echo ! empty( $options['hub_url'] ) ? esc_html( $options['hub_url'] ) : 'not set'; ?></code></p>
					<p>Send posts to hub: <strong><?php echo ! empty( $options['enable_sync'] ) ? 'On' : 'Off'; ?></strong></p>
					<p>Paywall enabled: <strong><?php echo ! empty( $options['enable_paywall'] ) ? 'On' : 'Off'; ?></strong></p>
					<p>Authori enabled: <strong><?php echo ! empty( $options['enable_authori'] ) ? 'On' : 'Off'; ?></strong><?php echo ! empty( $options['authori_auto'] ) ? ' (auto on publish)' : ''; ?></p>
					<p>Shared Secret set: <strong><?php echo ! empty( $options['site_secret'] ) ? 'Yes' : 'No'; ?></strong></p>
					<p>WordierPress Email: <strong><?php echo ! empty( $options['site_key'] ) ? esc_html( $options['site_key'] ) : 'not set'; ?></strong></p>
				</div>
			<?php elseif ( $tab === 'access' ) : ?>
				<div id="hw-access" class="card">
					<h2>Access</h2>
					<p>This allows access as an Author and enables most of the components here.</p>
					<form method="post" action="options.php">
						<?php
						settings_fields( 'hw_settings_group' );
						?>
						<table class="form-table">
							<tr>
								<th scope="row">WordierPress Email</th>
								<td><input type="email" class="regular-text" name="hubw_settings[site_key]" value="<?php echo esc_attr( $options['site_key'] ?? '' ); ?>" /></td>
							</tr>
							<tr>
								<th scope="row">Shared Secret</th>
								<td>
									<input type="password" class="regular-text" name="hubw_settings[site_secret]" value="<?php echo esc_attr( $options['site_secret'] ?? '' ); ?>" />
									<p class="description"><a href="https://wordierpress.com/get_shared_secret" target="_blank" rel="noreferrer">Get your shared secret</a></p>
								</td>
							</tr>
							<tr>
								<th scope="row">Hub URL</th>
								<td><input type="url" class="regular-text" name="hubw_settings[hub_url]" value="<?php echo esc_attr( $options['hub_url'] ?? '' ); ?>" placeholder="https://hub.example.com" /></td>
							</tr>
						</table>
						<?php submit_button( 'Save access settings' ); ?>
					</form>
				</div>
			<?php elseif ( $tab === 'update' ) : ?>
				<div id="updates" class="card">
					<h2>Update</h2>
					<p>Current version: <code><?php echo esc_html( HW_VERSION ); ?></code></p>
					<?php
					$last = get_option( 'hubw_update_last', array() );
					if ( ! empty( $last ) ) :
						$status_text = ! empty( $last['status'] ) ? $last['status'] : 'unknown';
						$time_text   = ! empty( $last['time'] ) ? $last['time'] : 'unknown';
						$msg_text    = ! empty( $last['msg'] ) ? $last['msg'] : '';
						?>
						<p>Last update: <strong><?php echo esc_html( $status_text ); ?></strong> at <?php echo esc_html( $time_text ); ?><?php echo $msg_text ? ' — ' . esc_html( $msg_text ) : ''; ?></p>
					<?php endif; ?>
					<p><a class="button button-primary" href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=hw_force_update' ), 'hw_force_update' ) ); ?>">Update hubWright now</a></p>
					<p><a class="button" href="https://wordierpress.com/hubwright/download">Download hubwright.zip</a></p>
				</div>
			<?php elseif ( $tab === 'paywall' ) : ?>
				<div id="paywall" class="card">
					<h2>Tiers</h2>
					<style>
						/* OAI-260104: balanced field widths for clarity. */
						#paywall .hw-tier-price { min-width: 90px; width: 100%; max-width: 140px; }
						#paywall .hw-tier-name { width: 260px; min-width: 220px; max-width: 320px; }
					</style>
					<form method="post" action="options.php">
						<?php settings_fields( 'hw_settings_group' ); ?>
						<table class="wp-list-table widefat striped">
							<thead>
								<tr>
									<th style="width:40px;">#</th>
									<th style="width:180px;">Name</th>
									<th>Description</th>
									<th style="width:140px;">$/month</th>
								</tr>
							</thead>
							<tbody>
								<tr>
									<td><strong>0</strong></td>
									<td><strong>Free</strong></td>
									<td><input type="text" name="hubw_settings[tiers_free_desc]" value="<?php echo esc_attr( isset( $options['tiers'][0]['desc'] ) ? $options['tiers'][0]['desc'] : '' ); ?>" class="regular-text" /></td>
									<td><input type="text" name="hubw_settings[tiers_free_price]" value="0" class="regular-text hw-tier-price" readonly /></td>
								</tr>
								<?php
								$paid = array_slice( isset( $options['tiers'] ) ? $options['tiers'] : array(), 1 );
								$default_names = array( 1 => 'One', 2 => 'Two', 3 => 'Three', 4 => 'Four', 5 => 'Five' ); // OAI-260104: friendly defaults.
								for ( $i = 1; $i <= 5; $i++ ) :
									$tier = isset( $paid[ $i - 1 ] ) ? $paid[ $i - 1 ] : array( 'name' => '', 'desc' => '', 'price' => '' );
									$display_name = $tier['name'] ?? '';
									if ( $display_name === '' && isset( $default_names[ $i ] ) ) {
										$display_name = $default_names[ $i ];
									}
									?>
								<tr>
									<td><?php echo intval( $i ); ?></td>
									<td><input type="text" name="hubw_settings[tiers_paid][<?php echo intval( $i ); ?>][name]" value="<?php echo esc_attr( $display_name ); ?>" class="regular-text hw-tier-name" /></td>
									<td><input type="text" name="hubw_settings[tiers_paid][<?php echo intval( $i ); ?>][desc]" value="<?php echo esc_attr( $tier['desc'] ?? '' ); ?>" class="regular-text" /></td>
									<td><input type="text" name="hubw_settings[tiers_paid][<?php echo intval( $i ); ?>][price]" value="<?php echo esc_attr( $tier['price'] ?? '' ); ?>" class="regular-text hw-tier-price" /></td>
								</tr>
								<?php endfor; ?>
							</tbody>
						</table>
						<?php submit_button( 'Update tiers' ); ?>
					</form>
				</div>
<?php elseif ( $tab === 'backups' ) : ?>
				<div class="card">
					<h2>Backups</h2>
					<p>This tool makes a General Full Backup (GFB) of all your WordPress content. This GFB downloads directly to your browser as a zip archive, or you may direct it to be stored instead elsewhere, including to your GFB area on wordierpress.com. This is a convenience for members. hubWright imposes no size limits whatsoever, nor does wordierpress.com. Storage fees may apply; see <a href="https://wordierpress.com/storage_fees" target="_blank" rel="noreferrer">wordierpress.com/storage_fees</a>.</p>
					<p>Note that this tool has no knowledge of nor any way to control whatever storage limits your WordPress host may be imposing upon you: abiding your limits and keeping clean your storage space use is your responsibility. Below is a list of any GFB files stored locally inside your WordPress.</p>
					<p>Recent tar.gz backups from <code>wp-content/hw-backups/</code>:</p>
					<?php if ( empty( $backups ) ) : ?>
						<p><em>No backups found.</em></p>
					<?php else : ?>
						<ul>
							<?php foreach ( $backups as $file ) : ?>
								<li><?php echo esc_html( basename( $file ) ); ?></li>
							<?php endforeach; ?>
						</ul>
					<?php endif; ?>
				</div>
			<?php elseif ( $tab === 'fans' ) : ?>
				<div class="card">
					<h2>Fans of Your Content</h2>
					<p>See members who follow your work across free and paid tiers; share updates via your Content Kiosk.</p>
					<h3>Free tier</h3>
					<p><em>Placeholder — list free followers from the hub.</em></p>
					<h3>Paid tier</h3>
					<p><em>Placeholder — list paid supporters and contact options.</em></p>
					<p style="margin-top:12px;"><strong>Content Kiosk note:</strong> you can broadcast a note to all fans via your Kiosk once wired.</p>
				</div>
			<?php elseif ( $tab === 'income' ) : ?>
				<div class="card">
					<h2>Income</h2>
					<p>Explore monetization: paywall conversions, offsite backup tiers, and migration add-ons.</p>
					<ul style="list-style:disc;margin-left:18px;">
						<li>Offer offsite partial/full backups over fixed retention windows for easy migration.</li>
						<li>Track paywall conversions and plan rev-shares for hubWright referrals.</li>
						<li>Bundle migration services (e.g., Substack import) with upsell options.</li>
						<li>Future: integrate payout reports and balance summaries here.</li>
					</ul>
					<p><em>Placeholder—wire billing and reporting endpoints here.</em></p>
				</div>
			<?php elseif ( $tab === 'kiosks' ) : ?>
				<div class="card">
					<h2>Kiosks</h2>
					<p>Purpose: your Content Kiosk is a public, shareable space to highlight posts, offers, and notes to fans; it can also relay updates to free/paid followers.</p>
					<p><em>Placeholder — wire Kiosk creation, customization, and publishing here.</em></p>
				</div>
			<?php elseif ( $tab === 'share' ) : ?>
				<div class="card">
					<h2>Share</h2>
					<form method="post" action="options.php">
						<?php settings_fields( 'hw_settings_group' ); ?>
						<table class="form-table">
							<tr>
								<th scope="row">Enable sharing of posts via hub</th>
								<td>
									<label>
										<input type="checkbox" name="hubw_settings[enable_sync]" value="1" <?php checked( ! empty( $options['enable_sync'] ) ); ?> />
										Share via hub
									</label>
									<p class="description">Toggle hub sharing; per-post “Share via hub” is in the editor sidebar.</p>
								</td>
							</tr>
						</table>
						<?php submit_button( 'Update settings' ); ?>
					</form>
					<p><strong>Status:</strong> Sharing is <strong><?php echo ! empty( $options['enable_sync'] ) ? 'enabled' : 'disabled'; ?></strong>.</p>
				</div>
			<?php elseif ( $tab === 'substack' ) : ?>
				<div class="card">
					<h2>Substack import</h2>
					<p>Import Substack exports (CSV) to migrate posts and subscribers into hubWright/WordPress.</p>
					<ul style="list-style:disc;margin-left:18px;">
						<li>Upload the Substack CSV export (posts and/or subscribers) to stage an import.</li>
						<li>Map fields to WordPress posts and users; optionally tag imported posts as paywalled.</li>
						<li>Future: schedule phased migration, preview conflicts, and send notification emails.</li>
					</ul>
					<form method="post" enctype="multipart/form-data" onsubmit="return false;" style="margin-top:12px;">
						<label for="hw_substack_csv">Substack CSV export</label><br>
						<input type="file" id="hw_substack_csv" name="hw_substack_csv" accept=".csv" disabled />
						<p><em>Placeholder — wire CSV parser and importer here.</em></p>
						<button class="button" type="button" disabled>Upload &amp; stage import</button>
					</form>
				</div>
			<?php elseif ( $tab === 'sys' ) : ?>
				<div class="card">
					<h2>Sys</h2>
					<p>Footprint and cleanup readiness for hubWright (all client-side artifacts use the hubw_ prefix).</p>
					<ul style="list-style:disc;margin-left:18px;">
						<li>Options: hubw_settings</li>
						<li>Transients: hubw_remote_version</li>
						<li>Meta keys: _hubw_paywall_gate, _hubw_authori_hash, _hubw_authori_stamped_at (legacy keys cleaned on save)</li>
						<li>Tables: <?php echo esc_html( $wpdb->prefix ); ?>hubw_shared_posts, <?php echo esc_html( $wpdb->prefix ); ?>hubw_authori_stamps</li>
						<li>Files: wp-content/hw-backups/, wp-content/hw-authori.log</li>
						<li>Plugin footprint: <strong><?php echo esc_html( size_format( $plugin_size ) ); ?></strong></li>
						<li>Stored update archives: <strong><?php echo esc_html( size_format( $archive_size ) ); ?></strong></li>
					</ul>
				</div>
			<?php elseif ( $tab === 'authori' ) : ?>
				<div class="card">
					<h2>Authori</h2>
					<p>Authori provides immutable proof of authorship on a tamper-proof registry—post hashes are signed to an append-only blockchain so anyone can verify provenance even if content is paywalled or moved.</p>
					<ul style="list-style:disc;margin-left:18px;">
						<li>Submit a post to Authori ledger.</li>
						<li>Provide readers a verification link.</li>
						<li>Blockchain proves submission date, time, content for copyright protection.</li>
					</ul>
					<form method="post" action="options.php" style="margin:12px 0;">
						<?php settings_fields( 'hw_settings_group' ); ?>
						<p>
							<label>
								<input type="checkbox" name="hubw_settings[enable_authori]" value="1" <?php checked( ! empty( $options['enable_authori'] ) ); ?> />
								Enable Authori checkbox in content Editor
							</label>
						</p>
						<?php submit_button( 'Update Authori settings', 'secondary', 'submit', false ); ?>
					</form>
					<?php if ( ! empty( $authori ) ) : ?>
						<h3>Recent stamps</h3>
						<ul>
							<?php foreach ( $authori as $entry ) : ?>
								<li>
									<strong><?php echo esc_html( isset( $entry['title'] ) ? $entry['title'] : 'Post ' . ( $entry['post_id'] ?? '' ) ); ?></strong>
									(<?php echo esc_html( $entry['timestamp'] ?? '' ); ?>)
									<?php if ( ! empty( $entry['url'] ) ) : ?>
										<a href="<?php echo esc_url( $entry['url'] ); ?>" target="_blank" rel="noreferrer">view</a>
									<?php endif; ?>
									<div style="font-size:12px;word-wrap:break-word;">Hash: <code><?php echo esc_html( $entry['hash'] ?? '' ); ?></code></div>
								</li>
							<?php endforeach; ?>
						</ul>
					<?php endif; ?>
					<p>
					<details style="margin:10px 0 14px 0;">
						<summary style="cursor:pointer;">How Authori works</summary>
						<div style="background:#f8fafc;border:1px solid #e2e8f0;padding:10px;border-radius:4px;white-space:normal;font-size:13px;margin-top:8px;font-family:-apple-system,BlinkMacSystemFont,Segoe UI,sans-serif;line-height:1.6;">
							<p style="margin:0 0 6px 0;"><strong>Here’s a pragmatic flow:</strong></p>
							<p style="margin:0 0 6px 0;">Normalize and hash the canonical post text (strip markup, normalize Unicode, trim whitespace), optionally include title + canonical URL + paywall flag; hash with SHA-256 and keep only the hash.</p>
							<p style="margin:0 0 6px 0;">Sign and log locally: build a signed JSON payload {hash, url, author_id, paywalled?, timestamp_utc, site_id} with a site key; store payload + signature + local audit log.</p>
							<p style="margin:0 0 6px 0;">Submit to Authori intake API (POST signed payload) and get an ingestion ID.</p>
							<p style="margin:0 0 6px 0;">Batch and anchor: Authori batches ingestions into a Merkle tree; the root is anchored on-chain (L2/mainnet) or durable store (e.g., Arweave/IPFS). The on-chain tx hash + Merkle proof link each post hash to the anchor.</p>
							<p style="margin:0 0 6px 0;">Persist proofs: save ingestion ID, Merkle proof, and on-chain tx hash to post meta for “stamped” status and verification links.</p>
							<p style="margin:0;">Verify path: recompute the post hash, fetch proof, and validate against the on-chain root.</p>
							<p style="margin:8px 0 0 0;">This keeps content private (only the hash travels), uses signatures to prove origin, and stores only small on-chain anchors for cost efficiency.</p>
						</div>
					</details>
					<?php if ( empty( $authori ) ) : ?>
						<p><em>No Authori stamps yet.</em></p>
					<?php endif; ?>
				</div>
			<?php elseif ( $tab === 'about' ) : ?>
				<div class="card">
					<h2>About</h2>
					<p><strong>United We Blog - WordierPress</strong><br>Indie voices.&nbsp;Share.&nbsp;Discover.&nbsp;Kindred authorship.&nbsp;Copyright protection.</p>
					<p><strong>What is WordierPress?</strong><br>A community of content creators connected through the hub for discovery, sharing, and collaboration.</p>
					<p><strong>What is hubWright?</strong><br>The WordPress tool that links your site to WordierPress—sync posts, gate content, stamp authorship, migrate from other platforms, and keep your readers connected.</p>
					<p><a href="https://wordierpress.com/about" target="_blank" rel="noreferrer">About WordierPress</a> &middot; <a href="https://wordierpress.com/contact" target="_blank" rel="noreferrer">Contact Us</a></p>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}

	private function list_backups() { // OAI-260103: enumerate stored backups.
		$dir = trailingslashit( WP_CONTENT_DIR ) . 'hw-backups';
		if ( ! is_dir( $dir ) ) {
			return array();
		}
		$files = glob( $dir . '/*.tar.gz' );
		rsort( $files );
		return $files;
	}

	private function list_push_log() { // OAI-260103: read push-update log tail.
		$log_file = trailingslashit( WP_CONTENT_DIR ) . 'hw-backups/push.log';
		if ( ! file_exists( $log_file ) ) {
			return array();
		}
		$lines = file( $log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES );
		return array_slice( array_reverse( $lines ), 0, 20 );
	}

	private function get_remote_version( $refresh = false ) {
		if ( $refresh ) {
			delete_transient( 'hubw_remote_version' );
		}
		$cached = get_transient( 'hubw_remote_version' );
		if ( $cached ) {
			if ( version_compare( HW_VERSION, $cached, '>' ) ) {
				// OAI-260103: Prevent stale cache showing older than current.
				set_transient( 'hubw_remote_version', HW_VERSION, MINUTE_IN_SECONDS );
				return HW_VERSION;
			}
			return $cached;
		}
		$response = wp_remote_get( HW_UPDATE_VERSION_URL ); // OAI-260104: fixed to wordierpress.com
		if ( is_wp_error( $response ) ) {
			return '';
		}
		if ( 200 !== wp_remote_retrieve_response_code( $response ) ) {
			return '';
		}
		$body = trim( wp_remote_retrieve_body( $response ) );
		if ( ! $body ) {
			return '';
		}
		set_transient( 'hubw_remote_version', $body, MINUTE_IN_SECONDS );
		return $body;
	}

	private function get_directory_size( $path ) { // OAI-260104: helper for footprint summary.
		if ( ! file_exists( $path ) ) {
			return 0;
		}
		$size = 0;
		$iterator = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $path, FilesystemIterator::SKIP_DOTS ) );
		foreach ( $iterator as $file ) {
			if ( $file->isFile() ) {
				$size += $file->getSize();
			}
		}
		return $size;
	}
}
