<?php
/**
 * Plugin Name: hubWright
 * Description: Connects WordPress sites to the collective hub and renders the reader bar.
 * Version: 20260104-131504 // OAI-260104
 * Author: Your Team
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'HW_VERSION', '20260104-131504' ); // OAI-260104: auto bump with UTC timestamp.
define( 'HW_PATH', plugin_dir_path( __FILE__ ) );
define( 'HW_URL', plugin_dir_url( __FILE__ ) );
define( 'HW_UPDATE_VERSION_URL', 'https://wordierpress.com/hubwright/version' ); // OAI-260104: lock updater to canonical source.
define( 'HW_UPDATE_PACKAGE_URL', 'https://wordierpress.com/hubwright/download' ); // OAI-260104: lock updater to canonical source.
// OAI-260103: HubWright bootstrap constants.

require_once HW_PATH . 'includes/class-hw-settings.php';
require_once HW_PATH . 'includes/class-hw-rest.php';
require_once HW_PATH . 'includes/class-hw-sync.php';
require_once HW_PATH . 'includes/class-hw-update.php';
require_once HW_PATH . 'includes/class-hw-authori.php';
require_once HW_PATH . 'includes/class-hw-paywall.php';
require_once HW_PATH . 'includes/class-hw-admin.php';

add_action(
	'plugins_loaded',
	function () {
		$settings = new HW_Settings();
		$rest     = new HW_REST( $settings );
		$sync     = new HW_Sync( $settings );
		$update   = new HW_Update( $settings );
		$authori  = new HW_Authori( $settings );
		$paywall  = new HW_Paywall( $settings );
		$admin    = new HW_Admin();

		$settings->register();
		$rest->register_routes();
		$sync->register();
		$update->register();
		$authori->register();
		$paywall->register();
		$admin->register();
	}
);

// OAI-260104: Install/maintain custom tables for hub submissions and Authori stamps.
register_activation_hook(
	__FILE__,
	function () {
		global $wpdb;
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$charset = $wpdb->get_charset_collate();

		$shared_table = $wpdb->prefix . 'hubw_shared_posts';
		$authori_table = $wpdb->prefix . 'hubw_authori_stamps';

		$shared_sql = "CREATE TABLE $shared_table (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			post_id BIGINT UNSIGNED NOT NULL,
			site_id VARCHAR(191) NOT NULL DEFAULT '',
			status VARCHAR(50) NOT NULL DEFAULT 'pending',
			payload LONGTEXT NULL,
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY post_id (post_id)
		) $charset;";

		$authori_sql = "CREATE TABLE $authori_table (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			post_id BIGINT UNSIGNED NOT NULL,
			hash VARCHAR(255) NOT NULL,
			stamped_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			meta LONGTEXT NULL,
			PRIMARY KEY  (id),
			KEY post_id (post_id),
			KEY hash (hash)
		) $charset;";

		dbDelta( $shared_sql );
		dbDelta( $authori_sql );
	}
);

// OAI-260104: Helpers to optimize or drop tables (not invoked automatically).
function hw_optimize_tables() {
	global $wpdb;
	$tables = array(
		$wpdb->prefix . 'hubw_shared_posts',
		$wpdb->prefix . 'hubw_authori_stamps',
	);
	foreach ( $tables as $table ) {
		$wpdb->query( "OPTIMIZE TABLE $table" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
	}
}

function hw_drop_tables() {
	global $wpdb;
	$tables = array(
		$wpdb->prefix . 'hubw_shared_posts',
		$wpdb->prefix . 'hubw_authori_stamps',
	);
	foreach ( $tables as $table ) {
		$wpdb->query( "DROP TABLE IF EXISTS $table" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
	}
}

// OAI-260104: Update archive helpers.
function hw_add_update_archive( $path, $version ) {
	if ( ! file_exists( $path ) ) {
		return;
	}
	$archives = get_option( 'hubw_update_archives', array() );
	$archives[] = array(
		'file'    => basename( $path ),
		'version' => $version,
		'time'    => current_time( 'mysql' ),
		'size'    => filesize( $path ),
	);
	// Trim to 5 newest and clean missing files.
	$archives = array_values(
		array_filter(
			$archives,
			function ( $entry ) {
				$dir  = trailingslashit( WP_CONTENT_DIR . '/hw-updates' );
				$file = $dir . $entry['file'];
				return file_exists( $file );
			}
		)
	);
	usort(
		$archives,
		function ( $a, $b ) {
			$at = isset( $a['time'] ) ? strtotime( $a['time'] ) : 0;
			$bt = isset( $b['time'] ) ? strtotime( $b['time'] ) : 0;
			return $bt <=> $at; // newest first.
		}
	);
	while ( count( $archives ) > 5 ) {
		$old = array_pop( $archives ); // remove oldest.
		$old_path = trailingslashit( WP_CONTENT_DIR . '/hw-updates' ) . $old['file'];
		if ( file_exists( $old_path ) ) {
			wp_delete_file( $old_path );
		}
	}
	update_option( 'hubw_update_archives', $archives );
}

function hw_get_update_archives() {
	$dir      = trailingslashit( WP_CONTENT_DIR . '/hw-updates' );
	wp_mkdir_p( $dir );
	$archives = get_option( 'hubw_update_archives', array() );

	// Merge in any on-disk zips.
	foreach ( glob( $dir . '*.zip' ) as $file ) {
		$name = basename( $file );
		$exists = false;
		foreach ( $archives as $entry ) {
			if ( isset( $entry['file'] ) && $entry['file'] === $name ) {
				$exists = true;
				break;
			}
		}
		if ( $exists ) {
			continue;
		}
		$version = '';
		if ( preg_match( '/hubwright-([0-9]{8}-[0-9]{6})(?:-[0-9]{8}-[0-9]{6})?\\.zip$/', $name, $m ) ) {
			$version = $m[1];
		}
		$archives[] = array(
			'file'    => $name,
			'version' => $version,
			'time'    => gmdate( 'Y-m-d H:i:s', filemtime( $file ) ),
			'size'    => filesize( $file ),
		);
	}

	$archives = array_values(
		array_filter(
			$archives,
			function ( $entry ) use ( $dir ) {
				$file = $dir . $entry['file'];
				return file_exists( $file );
			}
		)
	);
	usort(
		$archives,
		function ( $a, $b ) {
			$at = strtotime( $a['time'] ?? 'now' );
			$bt = strtotime( $b['time'] ?? 'now' );
			return $bt <=> $at;
		}
	);
	while ( count( $archives ) > 5 ) {
		$old = array_pop( $archives );
		$old_path = $dir . $old['file'];
		if ( file_exists( $old_path ) ) {
			wp_delete_file( $old_path );
		}
	}
	update_option( 'hubw_update_archives', $archives );
	return $archives;
}

add_action(
	'admin_post_hw_install_archive',
	function () {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}
		check_admin_referer( 'hw_install_archive' );
		if ( empty( $_POST['file'] ) ) {
			wp_safe_redirect( admin_url( 'admin.php?page=hw-admin&tab=update&hw_update=fail&hw_msg=' . rawurlencode( 'Missing archive file.' ) ) );
			exit;
		}
		$file = basename( sanitize_text_field( wp_unslash( $_POST['file'] ) ) );
		$path = trailingslashit( WP_CONTENT_DIR . '/hw-updates' ) . $file;
		if ( ! file_exists( $path ) ) {
			wp_safe_redirect( admin_url( 'admin.php?page=hw-admin&tab=update&hw_update=fail&hw_msg=' . rawurlencode( 'Archive not found.' ) ) );
			exit;
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		$skin     = new Automatic_Upgrader_Skin(
			array(
				'url' => admin_url( 'admin.php?page=hw-admin&tab=update' ),
			)
		);
		$upgrader = new Plugin_Upgrader( $skin );
		$result   = $upgrader->install( $path );
		if ( ! is_wp_error( $result ) ) {
			activate_plugin( 'hubwright/hubwright.php' );
			wp_safe_redirect( admin_url( 'admin.php?page=hw-admin&tab=update&hw_update=success' ) );
			exit;
		}

		wp_safe_redirect( admin_url( 'admin.php?page=hw-admin&tab=update&hw_update=fail&hw_msg=' . rawurlencode( $result->get_error_message() ) ) );
		exit;
	}
);

add_action(
	'admin_post_hw_delete_archive',
	function () {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}
		check_admin_referer( 'hw_delete_archive' );
		if ( empty( $_POST['file'] ) ) {
			wp_safe_redirect( admin_url( 'admin.php?page=hw-admin&tab=update' ) );
			exit;
		}
		$file = basename( sanitize_text_field( wp_unslash( $_POST['file'] ) ) );
		$path = trailingslashit( WP_CONTENT_DIR . '/hw-updates' ) . $file;
		if ( file_exists( $path ) ) {
			wp_delete_file( $path );
		}
		$archives = hw_get_update_archives();
		$archives = array_values(
			array_filter(
				$archives,
				function ( $entry ) use ( $file ) {
					return $entry['file'] !== $file;
				}
			)
		);
		update_option( 'hubw_update_archives', $archives );
		wp_safe_redirect( admin_url( 'admin.php?page=hw-admin&tab=update' ) );
		exit;
	}
);

// OAI-260103: Manual update trigger from admin with version check.
add_action(
	'admin_post_hw_update_now',
	function () {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}
		check_admin_referer( 'hw_update_now' );

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		$prev_version = HW_VERSION; // OAI-260103: track current version for rollback notice.

		$remote_version = '';
		$response       = wp_remote_get( HW_UPDATE_VERSION_URL ); // OAI-260104: fixed to wordierpress.com
		if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
			$remote_version = trim( wp_remote_retrieve_body( $response ) );
		}

		$package = HW_UPDATE_PACKAGE_URL; // OAI-260104: fixed to wordierpress.com
		$skin    = new Automatic_Upgrader_Skin(
			array(
				'url' => admin_url( 'admin.php?page=hw-admin' ),
			)
		); // OAI-260103: silent skin for clean redirect back to hubWright page.
		$upgrader = new Plugin_Upgrader( $skin );
		$dest     = WP_PLUGIN_DIR . '/hubwright';
		$backup   = $dest . '-backup';

		// OAI-260103: Prepare filesystem helpers.
		WP_Filesystem();
		global $wp_filesystem;

		// OAI-260104: Download package locally to archive directory.
		$tmp_zip = download_url( $package );
		if ( is_wp_error( $tmp_zip ) ) {
			// OAI-260104: fallback to any existing hubwright.zip in hw-updates.
			$fallback_zip = trailingslashit( WP_CONTENT_DIR . '/hw-updates' ) . 'hubwright.zip';
			if ( file_exists( $fallback_zip ) ) {
				$tmp_zip = $fallback_zip;
			} else {
				$redirect = add_query_arg(
					array(
						'hw_update' => 'fail',
						'hw_msg'    => rawurlencode( $tmp_zip->get_error_message() ),
					),
					admin_url( 'admin.php?page=hw-admin' )
				);
				wp_safe_redirect( $redirect );
				exit;
			}
		}

		$archive_dir  = trailingslashit( WP_CONTENT_DIR . '/hw-updates' );
		wp_mkdir_p( $archive_dir );
		$archive_name = 'hubwright-' . ( $remote_version ? $remote_version : HW_VERSION ) . '.zip';
		$archive_path = trailingslashit( $archive_dir ) . $archive_name;
		$wp_filesystem->delete( $archive_path );
		$copied = $wp_filesystem->copy( $tmp_zip, $archive_path, true, FS_CHMOD_FILE );
		if ( ! $copied && ! @rename( $tmp_zip, $archive_path ) ) { // OAI-260104: fallback to rename if copy fails.
			@unlink( $tmp_zip );
			hw_note_update_result( 'fail', 'Could not save update archive.' );
			$redirect = add_query_arg(
				array(
					'hw_update' => 'fail',
					'hw_msg'    => rawurlencode( 'Could not save update archive.' ),
				),
				admin_url( 'admin.php?page=hw-admin' )
			);
			wp_safe_redirect( $redirect );
			exit;
		}

		// OAI-260103: Create backup and clear current plugin to avoid destination conflicts.
		if ( is_dir( $backup ) ) {
			$wp_filesystem->delete( $backup, true );
		}
		if ( is_dir( $dest ) ) {
			@rename( $dest, $backup );
		}

		$result = $upgrader->install( $archive_path );
		@unlink( $tmp_zip );

		if ( ! is_wp_error( $result ) ) {
			activate_plugin( 'hubwright/hubwright.php' );
			hw_add_update_archive( $archive_path, $remote_version ? $remote_version : HW_VERSION );
			if ( is_dir( $backup ) ) {
				$wp_filesystem->delete( $backup, true );
			}
			hw_note_update_result( 'success', 'Installed from hubwright.zip' );
			$redirect = add_query_arg(
				array( 'hw_update' => 'success' ),
				admin_url( 'admin.php?page=hw-admin' )
			);
			wp_safe_redirect( $redirect );
			exit;
		}

		// OAI-260103: Roll back if install failed.
		if ( is_dir( $backup ) ) {
			$wp_filesystem->delete( $dest, true );
			@rename( $backup, $dest );
		}

		hw_note_update_result( 'fail', 'Error occurred; reverted to previous version.' );
		$redirect = add_query_arg(
			array(
				'hw_update' => 'fail',
				'hw_msg'    => rawurlencode( 'Error occurred; reverted to previous version.' ),
				'hw_prev'   => rawurlencode( $prev_version ),
			),
			admin_url( 'admin.php?page=hw-admin' )
		);
		wp_safe_redirect( $redirect );
		exit;
	}
);

// OAI-260104: Force update pulls whatever is at the download URL and installs immediately.
add_action(
	'admin_post_hw_force_update',
	function () {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}
		check_admin_referer( 'hw_force_update' );
		// Reuse update flow without version check.
		$_REQUEST['_wpnonce'] = wp_create_nonce( 'hw_update_now' );
		do_action( 'admin_post_hw_update_now' );
		exit;
	}
);

function hw_note_update_result( $status, $message = '' ) {
	update_option(
		'hubw_update_last',
		array(
			'status' => $status,
			'time'   => current_time( 'mysql' ),
			'msg'    => $message,
		)
	);
}

// OAI-260103: Expose update metadata so WP shows available updates on Plugins page.
add_filter(
	'site_transient_update_plugins',
	function ( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$remote_version = '';
		$response       = wp_remote_get( HW_UPDATE_VERSION_URL ); // OAI-260104: fixed to wordierpress.com
		if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
			$remote_version = trim( wp_remote_retrieve_body( $response ) );
		}

		if ( $remote_version && version_compare( $remote_version, HW_VERSION, '>' ) ) {
			$plugin = plugin_basename( __FILE__ );
			$transient->response[ $plugin ] = (object) array(
				'slug'        => 'hubwright',
				'plugin'      => $plugin,
				'new_version' => $remote_version,
				'url'         => 'https://wordierpress.com/hubwright',
				'package'     => 'https://wordierpress.com/hubwright/download',
			);
		}

		return $transient;
	}
);
